--[[---------------------------------------------------------------------------------------------------
											Collision
											
Description: Provides information about the player's armor durability
Author: Copyright 2008 João Cardoso (Jaliborc)
Contact: jaliborc@gmail.com
Dependencies: LibStub
License: GNU General Public License - Version 3 (<http://www.gnu.org/licenses/> for more information)
--]]------------------------------------------------------------------------------------------------

local Collision = LibStub:NewLibrary("Collision-1.3", 4)
if not Collision then
	return
end

local DurabilityFormat = "^"..gsub(DURABILITY_TEMPLATE, "%%d", "(%%d+)").."$"
local data

local function round(value, allow)
	if allow then
		return floor(value + 0.4)
	end
	return value
end


--[[ API ]]--

function Collision:GetArmorDurability(slot, precise)
	if type(slot) == 'string' then
		slot = GetInventorySlotInfo(slot)
	end
	
	if type(slot) ~= 'number' then
		error("Usage: Collicion:GetArmorDurability(slot): 'slot' must to be a number or contain one", 2)
	end
		
	if Collision.tooltip:SetInventoryItem("player", slot) then
		for i = 1, Collision.tooltip:NumLines() do
			local min, max = select(3, strfind(_G["CollisionTooltipTextLeft"..i]:GetText() or "", DurabilityFormat))
			if min and max then
				min = tonumber(min)
				max = tonumber(max)
				
				return min, max, round(min / max * 100, not precise)
			end
		end
	end
end

function Collision:GetOverallDurability(precise)
	local total, count = 0, 0
	for i = 1,18 do
		local percent = select(3, Collision:GetArmorDurability(i, true))
		if percent then
			total = total + percent
			count = count + 1
		end
	end

	return round(total / count, not precise)
end

function Collision:GetRepairCost()
	local total = 0
	for i = 1,18 do
		total = total + select(3, Collision.tooltip:SetInventoryItem("player", i))
	end
	return total
end

function Collision:PrintRepairCost(tooltip)
	if type(tooltip) ~= 'table' or type(tooltip.GetFrameType) ~= 'function' or tooltip:GetFrameType() ~= 'GameTooltip' then
		error("Usage: Collicion:ShowRepairCost(tooltip): variable provided isn't a tooltip", 3)
	end
	
	SetTooltipMoney(tooltip, Collision:GetRepairCost())
end

local function GetArmorStatus(value)
	data = {}
	local slot
	for i,info in pairs(INVENTORY_ALERT_STATUS_SLOTS) do
		if GetInventoryAlertStatus(i) == value then
			if info.slot ~= "Shield" or not OffhandHasWeapon() then
				slot = info.slot
			else
				slot = "OffWeapon"
			end
			data[i] = slot
		end
	end
	return data or {}
end

function Collision:IterateDamagedArmor() return pairs(GetArmorStatus(1)) end
function Collision:IterateBrokenArmor() return pairs(GetArmorStatus(2)) end


--[[ Embed ]]--

local mixins = {
	"GetArmorDurability", "GetOverallDurability",
	"GetRepairCost", "PrintRepairCost",
	"IterateDamagedArmor", "IterateBrokenArmor",
}

function Collision:Embed(target)
	for k,v in pairs(mixins) do
		target[v] = self[v]
	end
	self.embeds[target] = true
end

Collision.embeds = Collision.embeds or {}
for target in pairs(Collision.embeds) do
	Collision:Embed(target)
end


--[[ Final Touch ]]--

Collision.tooltip = Collision.tooltip or CreateFrame("GameTooltip", "CollisionTooltip", nil, "GameTooltipTemplate")
Collision.tooltip:SetOwner(WorldFrame, "ANCHOR_NONE")
setmetatable(Collision, { __call = Collision.Embed })